﻿using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using VA.PPMS.Context;


namespace VA.PPMS.CRM.Plugins
{
    public class ProviderServiceFieldMapping : IPlugin
    {
        private ITracingService tracingService;
        private const string PluginName = "ProviderServiceFieldMapping";
        private const string MessageCreate = "CREATE";
        private const string MessageUpdate = "UPDATE";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                //if (entity.LogicalName != "ppms_providerservice" || entity.LogicalName != "ppms_taxonomy" || entity.LogicalName != "ppms_caresite" || entity.LogicalName != "account" || entity.LogicalName != "ppms_serviceavailability")
                //    return;

                tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    //This plugin will be triggered by create/updates to different entities. We will handle the logic slightly different depending on which 
                    //message/event triggered the plugin. 

                    switch (entity.LogicalName)
                    {
                        //handle event first based on initiating entity. 
                        case "ppms_providerservice":
                            // handle event based on message type
                            switch (context.MessageName.ToUpper())
                            {
                                case MessageCreate:
                                    tracingService.Trace("Create of Provider Service: Update Provider Service Fields");
                                    UpdateProviderServiceFields(service, entity);
                                    break;

                                case MessageUpdate:

                                   // if (entity.Contains("ppms_updateprovservicefields") || entity.Contains("ppms_providerid") || entity.Contains("ppms_specialty") || entity.Contains("ppms_caresite"))
                                   // {
                                        tracingService.Trace("Update of Provider Service: Update Provider Service Fields");
                                        UpdateProviderServiceFields(service, entity);
                                   // }
                                    break;                               
                            }
                            break;
                        case "ppms_taxonomy":
                            switch (context.MessageName.ToUpper())
                            {
                                case MessageUpdate:

                                   // if (entity.Contains("ppms_codedspecialty") || entity.Contains("ppms_specialtycode"))
                                  //  {
                                        
                                        //Lookup specialty and retrieve Prov Service Entity Reference
                                        using (var svc = new PpmsContext(service))
                                        {
                                            var specialty = svc.ppms_taxonomySet.First(i => i.Id == entity.Id);
                                            var provServices =
                                                svc.ppms_providerserviceSet.Where(i =>
                                                    i.ppms_specialty.Id == specialty.Id);
 
                                            if (provServices != null)
                                            {
                                                var provServicesList = provServices.ToList();
                                                if (provServicesList.Any())
                                                    tracingService.Trace(
                                                        "Update of Specialty: Map Specialty to Provider Service Fields");
                                                MapSpecialtyToProviderServiceFields(service, provServicesList);
                                            }
                                        }
                                        
                                   // }
                                    break;
                            }
                            break;

                        case "ppms_caresite":
                            switch (context.MessageName.ToUpper())
                            {
                                case MessageUpdate:

                                   // if (entity.Contains("ppms_address_line1") || entity.Contains("ppms_address_city") || entity.Contains("ppms_address_state") || entity.Contains("ppms_address_postalcode") || entity.Contains("ppms_address_latitude") || entity.Contains("ppms_address_longitude"))
                                   // {
                                        //Lookup care site and retrieve Prov Service Entity Reference
                                        using (var svc = new PpmsContext(service))
                                        {
                                            var careSite = svc.ppms_caresiteSet.First(i => i.Id == entity.Id);
                                            var provServices =
                                                 svc.ppms_providerserviceSet.Where(i =>
                                                     i.ppms_caresite.Id == careSite.Id);
                                            if (provServices != null)
                                            {
                                                var provServicesList = provServices.ToList();
                                                if (provServicesList.Any())
                                                    tracingService.Trace(
                                                        "Update of Care Site: Map Care Site to Provider Service Fields");
                                                MapCareSiteToProviderServiceFields(service, provServicesList);
                                            }
                                        }

                                   // }
                                    break;
                            }
                            break;

                        case "account":
                            switch (context.MessageName.ToUpper())
                            {
                                case MessageUpdate:
                                    tracingService.Trace("Update of Provider: Update Provider Service Fields");
                                    
                                    //Retrieve Provider record to 

                                   // if (entity.Contains("Name") || entity.Contains("ppms_gender") || entity.Contains("ppms_individualisacceptingnewpatients") || entity.Contains("ppms_primarycareprovideracceptingva") || entity.Contains("ppms_PrimaryCarePhysician") || entity.Contains("ppms_qualityrankingtotalscore") || entity.Contains("ppms_ProviderIdentifier"))
                                    //{
                                        
                                        //Lookup Provider and retrieve Prov Service Entity Reference
                                        using (var svc = new PpmsContext(service))
                                        {
                                            tracingService.Trace("Retrieve the Provider record");
                                            var provider = svc.AccountSet.First(i => i.Id == entity.Id);
                                            tracingService.Trace("Looking up related Provider Service records");
                                            var provServices =
                                                 svc.ppms_providerserviceSet.Where(i =>
                                                     i.ppms_ProviderId.Id == provider.Id);
                                            if (provServices != null)
                                            {
                                                var provServicesList = provServices.ToList();
                                                if (provServicesList.Any())
                                                    tracingService.Trace(
                                                        "Update: Map Provider to Provider Service Fields");
                                                MapProviderToProviderServiceFields(service, provServicesList);
                                            }
                                        }

                                   // }
                                    break;
                            }
                            break;

                        case "ppms_serviceavailability":
                            switch (context.MessageName.ToUpper())
                            {                              
                                case MessageUpdate:

                                    //Only run on update because the workflow needs to set the name after create. 
                                   // if (entity.Contains("ppms_name"))
                                   // {
                                        //Lookup care site and retrieve Prov Service Entity Reference
                                        using (var svc = new PpmsContext(service))
                                        {
                                            var serviceAvailability =
                                                svc.ppms_serviceavailabilitySet.First(i => i.Id == entity.Id);
                                            var provService = svc.ppms_providerserviceSet.FirstOrDefault(i =>
                                                i.Id == serviceAvailability.ppms_ProviderServiceId.Id);

                                            if (provService != null)
                                                tracingService.Trace("Update of Service Availability: Map Service Availability to Provider Service Fields");
                                            MapServiceAvaialbilityToProviderServiceFields(service, provService);
                                        }
                                   // }
                                    break;
                            }
                            break;

                        default:
                            tracingService.Trace("Valid event message/entity not found");
                            break;
                    }
                    
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            tracingService.Trace("Done");
        }
      
        private void UpdateProviderServiceFields(IOrganizationService service, Entity entity)
        {           
            using (var svc = new PpmsContext(service))
            {
                var updateProvService = new ppms_providerservice
                {
                    Id = entity.Id
                };

                // Get Provider Service and needed attributes      
                //var provService = service.Retrieve("ppms_providerservice", entity.Id, new ColumnSet(new string[] { "ppms_providerid", "ppms_specialty", "ppms_caresite"}));
                var provService = svc.ppms_providerserviceSet.First(i => i.Id == entity.Id);
                tracingService.Trace("Provider Service retrieved: {0}", provService.Id.ToString());

                //Get Care Site Entity Ref. 
                //var careSiteReference = provService.GetAttributeValue<EntityReference>("ppms_caresite");
                var careSiteReference = provService.ppms_caresite;

                if (careSiteReference != null)
                {
                    tracingService.Trace("Care Site Reference found: {0}", careSiteReference.Id.ToString());

                    
                    var careSite = svc.ppms_caresiteSet.First(i => i.Id == careSiteReference.Id);
                    tracingService.Trace("Care Site found: {0}", careSite.Id.ToString());
                    updateProvService.ppms_caresiteaddress = careSite.ppms_address_line1;
                    updateProvService.ppms_caresitecity = careSite.ppms_address_city;
                    updateProvService.ppms_caresitestateprovince = careSite.ppms_statename;
                    updateProvService.ppms_caresitezipcode = careSite.ppms_address_postalcode;
                    updateProvService.ppms_CareSiteAddressLatitude = careSite.ppms_address_latitude;
                    updateProvService.ppms_CareSiteAddressLongitude = careSite.ppms_address_longitude;

                    
                }
                else
                {
                    tracingService.Trace("Care Site Reference not found");
                }

                //Get the Provider Reference. 
                //var providerReference = provService.GetAttributeValue<EntityReference>("ppms_providerid");
                var providerReference = provService.ppms_ProviderId;
                if (providerReference != null)
                {
                    tracingService.Trace("Provider Reference found: {0}", providerReference.Id.ToString());

                    var provider = svc.AccountSet.First(i => i.Id == providerReference.Id);
                    updateProvService.ppms_providername = provider.Name;
                    updateProvService.ppms_providergender = provider.ppms_gender;
                    updateProvService.ppms_provideracceptingnewpatients = provider.ppms_individualisacceptingnewpatients;
                    updateProvService.ppms_provideracceptingva = provider.ppms_primarycareprovideracceptingva;
                    updateProvService.ppms_providerisprimarycare = provider.ppms_PrimaryCarePhysician;                   
                    updateProvService.ppms_qualityrankingtotalscore = provider.ppms_qualityrankingtotalscore;
                    updateProvService.ppms_provideridentifer = provider.ppms_ProviderIdentifier;
                    
                }
                else
                {
                    tracingService.Trace("Provider Reference not found");
                }

                //Get Specialty Entity Ref. 
                //var specialtyReference = provService.GetAttributeValue<EntityReference>("ppms_specialty");
                var specialtyReference = provService.ppms_specialty;
                if (specialtyReference != null)
                {
                    tracingService.Trace("Specialty Reference found: {0}", specialtyReference.Id.ToString());

                    var specialty = svc.ppms_taxonomySet.First(i => i.Id == specialtyReference.Id);

                    updateProvService.ppms_specialtynametext = specialtyReference.Name;
                    updateProvService.ppms_specialtycode = specialty.ppms_SpecialtyCode;
               
                }
                else
                {
                    tracingService.Trace("Specialty Reference not found");
                }

                //Get Working Hours
                var workingHours = svc.ppms_serviceavailabilitySet
                    .Where(sa => sa.StatusCode.Value == (int)ppms_serviceavailability_StatusCode.Active &&
                                 sa.ppms_ProviderServiceId.Id == entity.Id);

                //Add to list and sort by the Day of week. 
                if (workingHours != null)
                {
                    var workingHoursList = workingHours.ToList();
                    var workingHoursListSorted = workingHoursList.OrderBy(wh => wh.ppms_dayofweek.Value);

                    string hours = String.Empty;
 
                    foreach (var workingHour in workingHoursListSorted)
                    {
                        if (workingHour.Equals(workingHoursListSorted.Last()))
                        {
                            hours += workingHour.ppms_name;
                        }
                        else
                        {
                            hours += workingHour.ppms_name;
                            hours += ", ";
                        }
                    }

                    updateProvService.ppms_workhours = hours;

                    //Update the Provider Service record with new values. 
                    service.Update(updateProvService);
                    tracingService.Trace("Provider Service Fields Updated With All References");
                }
            }
        }

        private void MapSpecialtyToProviderServiceFields(IOrganizationService service, List<ppms_providerservice> provServicesList)
        {
            using (var svc = new PpmsContext(service))
            {
                foreach (var provService in provServicesList) {

                    var updateProvService = new ppms_providerservice
                    {
                        Id = provService.Id
                    };

                    // Get Specialty attributes      
                    var specialtyReference = provService.ppms_specialty;                    
                    var specialty = svc.ppms_taxonomySet.First(i => i.Id == specialtyReference.Id);
                    updateProvService.ppms_specialtynametext = specialtyReference.Name;
                    updateProvService.ppms_specialtycode = specialty.ppms_SpecialtyCode;
                   
                    //Update the Provider Service record with new values. 
                    service.Update(updateProvService);
                    tracingService.Trace("Provider Service: {0} Updated with Specialty Fields", provService.Id.ToString());
                }
            }
        }

        private void MapCareSiteToProviderServiceFields(IOrganizationService service, List<ppms_providerservice> provServicesList)
        {
            using (var svc = new PpmsContext(service))
            {
                foreach (var provService in provServicesList)
                {
                    var updateProvService = new ppms_providerservice
                    {
                        Id = provService.Id
                    };

                    //Get Care Site and needed attributes 
                    var careSiteReference = provService.ppms_caresite;
                    var careSite = svc.ppms_caresiteSet.First(i => i.Id == careSiteReference.Id);
                    updateProvService.ppms_caresiteaddress = careSite.ppms_address_line1;
                    updateProvService.ppms_caresitecity = careSite.ppms_address_city;
                    updateProvService.ppms_caresitestateprovince = careSite.ppms_statename;
                    updateProvService.ppms_caresitezipcode = careSite.ppms_address_postalcode;
                    updateProvService.ppms_CareSiteAddressLatitude = careSite.ppms_address_latitude;
                    updateProvService.ppms_CareSiteAddressLongitude = careSite.ppms_address_longitude;
                    
                    //Update the Provider Service record with new values. 
                    service.Update(updateProvService);
                    tracingService.Trace("Provider Service: {0} Updated with Care Site Fields", provService.Id.ToString());
                }
            }
        }

        private void MapProviderToProviderServiceFields(IOrganizationService service, List<ppms_providerservice> provServicesList)
        {
            using (var svc = new PpmsContext(service))
            {
                foreach (var provService in provServicesList)
                {
                    var updateProvService = new ppms_providerservice
                    {
                        Id = provService.Id
                    };

                    //Get the Provider Reference. 
                    var providerReference = provService.ppms_ProviderId;
                    var provider = svc.AccountSet.First(i => i.Id == providerReference.Id);
                    updateProvService.ppms_providername = provider.Name;
                    updateProvService.ppms_providergender = provider.ppms_gender;
                    updateProvService.ppms_provideracceptingnewpatients = provider.ppms_individualisacceptingnewpatients;
                    updateProvService.ppms_provideracceptingva = provider.ppms_primarycareprovideracceptingva;
                    updateProvService.ppms_providerisprimarycare = provider.ppms_PrimaryCarePhysician;
                    updateProvService.ppms_qualityrankingtotalscore = provider.ppms_qualityrankingtotalscore;
                    updateProvService.ppms_provideridentifer = provider.ppms_ProviderIdentifier;
                    
                    //Update the Provider Service record with new values. 
                    service.Update(updateProvService);
                    tracingService.Trace("Provider Service: {0} Updated with Provider Fields", provService.Id.ToString());
                }
            }
        }

        private void MapServiceAvaialbilityToProviderServiceFields(IOrganizationService service, ppms_providerservice provService)
        {
            using (var svc = new PpmsContext(service))
            {
                var updateProvService = new ppms_providerservice
                {
                    Id = provService.Id
                };

                //Get Working Hours
                var workingHours = svc.ppms_serviceavailabilitySet
                    .Where(sa => sa.StatusCode.Value == (int)ppms_serviceavailability_StatusCode.Active &&
                                 sa.ppms_ProviderServiceId.Id == provService.Id);

                //Add to list and sort by day of week.
                if (workingHours != null)
                {
                    var workingHoursList = workingHours.ToList();
                    var workingHoursListSorted = workingHoursList.OrderBy(wh => wh.ppms_dayofweek.Value);

                    string hours = String.Empty;
               
                    foreach (var workingHour in workingHoursListSorted)
                    {
                        if (workingHour.Equals(workingHoursListSorted.Last()))
                        {
                            hours += workingHour.ppms_name;
                        }
                        else
                        {
                            hours += workingHour.ppms_name;
                            hours += ", ";
                        }
                    }
                    
                    updateProvService.ppms_workhours = hours;

                    //Update the Provider Service record with new values. 
                    service.Update(updateProvService);
                    tracingService.Trace("Provider Service Fields Updated with Work Hours");
                }
            }
        }
    }
}
